/* ***************************************************************************+
 * ITX package (cnrg.itx) for telephony application programming.              *
 * Copyright (c) 1999  Cornell University, Ithaca NY.                         *
 * A copy of the license is distributed with this package.  Look in the docs  *
 * directory, filename GPL.  Contact information: bergmark@cs.cornell.edu     *
 ******************************************************************************/
package cnrg.itx.ds;

import java.util.*;
import java.text.*;
import cnrg.itx.ds.LocationList;

/**
 * This class represents a list of locations with user-specified time interval.
 */
public class CustomLocation extends LocationList {
	
	private Calendar m_startTime = null;
	private Calendar m_endTime = null;
	
	private static final String HDR_STARTTIME = "starttime=";
	private static final String HDR_ENDTIME = "endtime=";
	private static final String DELIMITER = ";";
	private static final int OFFSET_HDR_STARTTIME = HDR_STARTTIME.length();
	private static final int OFFSET_HDR_ENDTIME = HDR_ENDTIME.length();
		
	/**
	 * Constructor
	 * @param startTime specifying the starting time
	 * @param endTime specifying the ending time
	 * @param lstLocation list of location
	 */ 
	public CustomLocation(Calendar startTime, Calendar endTime, LocationList lstLocation){
		super(lstLocation);
		m_startTime = startTime;
		m_endTime = endTime;
	}

	/**
	 * Constructor for formatted entry string retrieved from the directory database.
	 * @param strRawString formatted entry string
	 */
	public CustomLocation(String strRawString){
		
		String field;
		Date tmpdate = new Date();
		StringTokenizer st = new StringTokenizer(strRawString, DELIMITER);
		int nCount = st.countTokens();
		
		field = st.nextToken();
		tmpdate.setTime(Long.parseLong(field.substring(OFFSET_HDR_STARTTIME)));
		m_startTime = Calendar.getInstance();
		m_startTime.setTime(tmpdate);
		
		field = st.nextToken();
		tmpdate.setTime(Long.parseLong(field.substring(OFFSET_HDR_ENDTIME)));
		m_endTime = Calendar.getInstance();
		m_endTime.setTime(tmpdate);


		for (int index=0; index<nCount-2; index++) {
			field = st.nextToken();
			((LocationList)this).add(new Location(field));
		}
	}
	
	/**
	 * Check if this custom location object is currently valid for use
	 * @return True if the custom location is valid for use at the current time
	 */
	public boolean currentValid() {
		
		Calendar cal = Calendar.getInstance();
		if ( (cal.after(m_startTime) && cal.before(m_endTime)) || cal.equals(m_startTime) || cal.equals(m_endTime) )
			return true;
		return false;
	}
		
	/**
	 * Get StartTime
	 * @return Calendar object representing a starting time
	 */ 
	public Calendar getStartTime(){
		return m_startTime;
	}

	/**
	 * Get EndTime
	 * @return Calender object representing an ending time
	 */ 
	public Calendar getEndTime(){
		return m_endTime;
	}
	
	/**
	 * Get Location list
	 * @return LocationList object
	 */
	public LocationList getLocationList(){
		return (LocationList)this;
	}
	
	/**
	 * Set StartTime
	 * @param newStartTime specifying a new starting time
	 */
	public void setStartTime(Calendar newStartTime){
		m_startTime = newStartTime;
	}

	/**
	 * Set EndTime
	 * @param newEndTime specifying a new end time
	 */
	public void setEndTime(Calendar newEndTime){
		m_endTime = newEndTime;
	}
	
	/**
	 * Set the Location list to the given new location list
	 * @param newLocList the given new location list (if newLocList is null or empty, it does nothing and exits)
	 */
	public void setLocationList(LocationList newLocList) {
		if (newLocList != null && newLocList.count() > 0) {
			((LocationList)this).removeAll();
			((LocationList)this).add(newLocList.first());
			Location tmploc = newLocList.next();
			while ( tmploc != null ) {
				((LocationList)this).add(tmploc);
				tmploc = newLocList.next();
			}
		}
	}
	
	/**
	 * Convert into formatted entry string to be stored in the directory database
	 * @return String to represents the formatted entry string to be stored in the directory database
	 */
	public String toString(){
		StringBuffer str = new StringBuffer(HDR_STARTTIME + String.valueOf(m_startTime.getTime().getTime()) + DELIMITER + HDR_ENDTIME + String.valueOf(m_endTime.getTime().getTime()) );
		
		Location loc = ((LocationList)this).first();
		str.append(DELIMITER + loc.toString());
		while ( (loc=((LocationList)this).next()) != null){
			str.append(DELIMITER + loc.toString());
		}
		return str.toString();
	}
}
